using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using HIPS.CommonSchemas;
using HIPS.PcehrSchemas;
using HIPS.Web.BusinessLogic.AssistedRegistration;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Web;
using HIPS.Web.Data.Hips;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Conversion.AssistedRegistration;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.ViewModels.AssistedRegistration;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// <see cref="Controller"/> for the PCEHR Assisted Registration functionality.
    /// </summary>
    public class AssistedRegistrationController : Controller
    {
        private IAssistedRegistrationReferenceRepository ARReferenceRepository { get; set; }
        private IHospitalRepository HospitalRepository { get; set; }
        private IPatientsWithoutPcehrRepository PatientsWithoutPcehrRepository { get; set; }
        private IAssistedRegistrationService AssistedRegistrationService { get; set; }
        private ISettingsRepository SettingsRepository { get; set; }

        private readonly List<Setting> Settings;

        private string DefaultHospitalCodeSystem
        {
            get { return Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem); }
        }

        //public AssistedRegistrationController() 
        //    : this(new CachedAssistedRegistrationReferenceRepository(new WebsiteDbRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))), 
        //           new CachedHospitalRepository(new HospitalRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))), 
        //           new CachedPatientsWithoutPcehrRepository(new PatientWebServiceRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))), 
        //           new ValidatedAssistedRegistrationService(new AssistedRegistrationService()),
        //           new CachedSettingsRepository(new WebsiteDbRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))))
        //{
        //}

        public AssistedRegistrationController(IAssistedRegistrationReferenceRepository arReferenceRepository, IHospitalRepository hospitalRepository, IPatientsWithoutPcehrRepository patientRepository, IAssistedRegistrationService registrationService, ISettingsRepository settingsRepository)
        {
            ARReferenceRepository = arReferenceRepository;
            HospitalRepository = hospitalRepository;
            PatientsWithoutPcehrRepository = patientRepository;
            AssistedRegistrationService = registrationService;
            SettingsRepository = settingsRepository;
            Settings = settingsRepository.GetSettings();
        }

        [HttpGet]
        public ActionResult Unregistered(string hospitalCode)
        {
            // Convert blank hospital codes to null
            if (String.IsNullOrWhiteSpace(hospitalCode))
            {
                hospitalCode = null;
            }

             // Create ViewModel
            UnregisteredViewModel viewModel = new UnregisteredViewModel { HospitalCode = hospitalCode };

            // Load reference data
            List<PcehrDataStore.Schemas.Hospital> hospitals = HospitalRepository.GetHospitals(DefaultHospitalCodeSystem);

            // Update ViewModel with reference data
            viewModel.Hospitals = hospitals.ToSelectListItems(h => h.GetCode(DefaultHospitalCodeSystem), h => h.Name);

            // Don't allow load for all patients
            if (hospitalCode == null)
            {
                // Render view
                return View(viewModel);
            }

            // Load patients without PCEHR
            PatientWithoutPcehrResponse response = PatientsWithoutPcehrRepository.ListPatientsWithoutPcehr(DefaultHospitalCodeSystem, hospitalCode);

            // Ensure loading was successful
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                // TODO: Verify appropriate error format / okay to use .ToString()?
                throw new ApplicationException(String.Format("Error encountered while retrieving patients without PCEHR. ({0})", response.HipsResponse.ToString())); 
            }

            // Update ViewModel with patients
            viewModel.LoadFromPatientsWithoutPcehr(response.PatientWithoutPcehrList);

            // Render view
            return View(viewModel);
        }

        [HttpGet]
        public ActionResult Register(string patientId, string hospitalCode)
        {
            // Load Reference Data
            List<Sex> sexes = ARReferenceRepository.GetSexes();
            List<IndigenousStatus> indigenousStatuses = ARReferenceRepository.GetIndigenousStatuses();
            List<IdentityVerificationMethod> identityVerificationMethods = ARReferenceRepository.GetIdentityVerificationMethods();
            List<Hospital> hospitals = LoadHospitals(DefaultHospitalCodeSystem);

            return Register(patientId, hospitalCode, false);
        }

        [HttpGet]
        public ActionResult RegisterDependant(string patientId, string hospitalCode)
        {
            // Load Reference Data
            List<Sex> sexes = ARReferenceRepository.GetSexes();
            List<IndigenousStatus> indigenousStatuses = ARReferenceRepository.GetIndigenousStatuses();
            List<IdentityVerificationMethod> identityVerificationMethods = ARReferenceRepository.GetIdentityVerificationMethods();
            List<Hospital> hospitals = LoadHospitals(DefaultHospitalCodeSystem);


            return Register(patientId, hospitalCode, true);
        }
        
        //TODO: Consider base64 or similar for patientId to obscure.
        private ActionResult Register(string patientId, string hospitalCode, bool asDependant)
        {
            // Load data
            List<Sex> sexes = ARReferenceRepository.GetSexes();
            List<IndigenousStatus> indigenousStatuses = ARReferenceRepository.GetIndigenousStatuses();
            List<IdentityVerificationMethod> identityVerificationMethods = ARReferenceRepository.GetIdentityVerificationMethods();
            List<Hospital> hospitals = LoadHospitals(DefaultHospitalCodeSystem);
            List<MedicareConsent> medicareConsents = ARReferenceRepository.GetMedicareConsents();

            // If patient indicated attempt to load
            if (!String.IsNullOrWhiteSpace(patientId))
            {
                List<ValidatedPatient> patients = PatientsWithoutPcehrRepository.GetPatientsWithoutPcehr(DefaultHospitalCodeSystem, sexes);
                
                // Filter on IHI + Hospital in case patient in multiple hospitals
                ValidatedPatient patient = patients.SingleOrDefault(p => p.VerifiedIhi.Ihi == patientId && p.CurrentLocation.Hospital.HospitalFacilityCode == hospitalCode);

                // If patient found, use for view
                if (patient != null)
                {
                    if (asDependant)
                    {
                        return View("RegisterDependant", new RegisterViewModel(patient, sexes, indigenousStatuses, identityVerificationMethods, hospitals, medicareConsents));
                    }
                    else
                    {
                        return View("Register", new RegisterViewModel(patient, sexes, indigenousStatuses, identityVerificationMethods, hospitals, medicareConsents));
                    }
                }
            }

            // NB: Configuration point if disabling visitor functionality

            // No patient loaded, render blank form
            if (asDependant)
            {
                var viewModel = new RegisterViewModel(sexes, indigenousStatuses, identityVerificationMethods, hospitals, medicareConsents);
                viewModel.Applicant.ShowDvaFileNumber = false; // TODO: Improve way we stop DVA being an option for dependant applicants.
                return View("RegisterDependant", viewModel);
            }
            else
            {
                return View("Register", new RegisterViewModel(sexes, indigenousStatuses, identityVerificationMethods, hospitals, medicareConsents));
            }
        }

        [HttpPost]
        public ActionResult Register(RegisterViewModel registration)
        {
            return Register(registration, false);
        }

        [HttpPost]
        public ActionResult RegisterDependant(RegisterViewModel registration)
        {
            return Register(registration, true);
        }

        [HttpPost]
        private ActionResult Register(RegisterViewModel registration, bool asDependant)
        {
            // As returning JSON ensure the request was AJAX.
            // TODO: Support non-ajax requests to this method (currently not enabled by client).
            // NB: Shouldn't be hard, priorities currently deferred (MM)
            if (!Request.IsAjaxRequest())
            {
                return Content("JavaScript is required for this functionality.");
            }

            // Check binding validation errors
            if (!ModelState.IsValid)
            {
                return Json(new { Echo = registration, Errors = ModelState.ToErrorDictionary() });
            }

            // Load Data
            List<Sex> sexes = ARReferenceRepository.GetSexes();
            List<IndigenousStatus> indigenousStatuses = ARReferenceRepository.GetIndigenousStatuses();
            List<IdentityVerificationMethod> identityVerificationMethods = ARReferenceRepository.GetIdentityVerificationMethods();
            List<Hospital> hospitals = LoadHospitals(DefaultHospitalCodeSystem);
            List<MedicareConsent> medicareConsentOptions = ARReferenceRepository.GetMedicareConsents();
            List<IvcDeliveryMethod> ivcDeliveryMethods = ARReferenceRepository.GetIvcDeliveryMethods();

            // If identifying a pre-validated applicant
            if (!String.IsNullOrWhiteSpace(registration.Applicant.Ihi))
            {
                List<ValidatedPatient> patients = PatientsWithoutPcehrRepository.GetPatientsWithoutPcehr(DefaultHospitalCodeSystem, sexes);
                // Filter on IHI and Hospital in case patient in multiple hospitals
                ValidatedPatient patient = patients.SingleOrDefault(p => p.VerifiedIhi.Ihi == registration.Applicant.Ihi && p.CurrentLocation.Hospital.HospitalFacilityCode == registration.HospitalCode);

                // Ensure patient found
                if (patient == null)
                {
                    ModelState.AddModelError(String.Empty, "Could not find the pre-validated patient record.");
                    return Json(new { Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }

                if (asDependant)
                {
                    RegisterResponse response = AssistedRegistrationService.RegisterDependant(
                        patient,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.Representative.ToPersonDemographicModel(sexes),
                        registration.ConsentDeclared,
                        registration.MedicareConsents.GetMedicareConsents(medicareConsentOptions),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
                else
                {
                    RegisterResponse response = AssistedRegistrationService.RegisterApplicant(
                        patient,
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.MedicareConsents.GetMedicareConsents(medicareConsentOptions),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }

            }
            else // A "visitor" applicant
            {
                // NB: Configuration point for "Visitor" functionality?

                if (asDependant)
                {
                    RegisterResponse response = AssistedRegistrationService.RegisterDependant(
                        registration.Applicant.ToPersonDemographicModel(sexes),
                        registration.GetHospital(hospitals),
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.Representative.ToPersonDemographicModel(sexes),
                        registration.ConsentDeclared,
                        registration.MedicareConsents.GetMedicareConsents(medicareConsentOptions),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
                else
                {
                    RegisterResponse response = AssistedRegistrationService.RegisterApplicant(
                        registration.Applicant.ToPersonDemographicModel(sexes),
                        registration.GetHospital(hospitals),
                        registration.ConsentDeclared,
                        registration.Applicant.GetIndigenousStatus(indigenousStatuses),
                        registration.MedicareConsents.GetMedicareConsents(medicareConsentOptions),
                        registration.GetIdentityVerificationMethod(identityVerificationMethods),
                        registration.GetIvcDeliveryMethodDecision(ivcDeliveryMethods),
                        this.GetCurrentUserDetails(),
                        registration.PatientMasterId);

                    return Json(new { Response = response, Echo = registration, Errors = ModelState.ToErrorDictionary() });
                }
            }
        }

        #region Cache Lookups

        //TODO: Refactor remove
        private List<Hospital> LoadHospitals(string hospitalCodeSystem)
        {
            List<PcehrDataStore.Schemas.Hospital> hospitals = HospitalRepository.GetHospitals(DefaultHospitalCodeSystem);
            return hospitals.Select(h => new Hospital(h.GetCode(hospitalCodeSystem), hospitalCodeSystem, h.Name, h.HpiO, h.HpioName))
                .ToList();
        }

        #endregion
    }
}